module subrotinas
  contains

  subroutine mapa(x0, y0, x1, y1)
    use constantes
    implicit none
    real(8), intent(in) :: x0, y0
    real(8), intent(out) :: x1, y1
    real(8) :: tn
    
    tn = theta - phi/(x0**2 + y0**2 + 1.d0)
    
    x1 = 1.d0 + ux*(x0*cos(tn) - y0*sin(tn))
    y1 = uy*(x0*sin(tn) + y0*cos(tn))
    
    return
  end subroutine mapa

  subroutine jacobian_matrix(dm, x, y)
    use constantes
    implicit none
    real(8), intent(in) :: x, y
    real(8), dimension(2,2), intent(out) :: dm
    real(8) :: tn, dtn_dx, dtn_dy

    tn = theta - phi/(x**2 + y**2 + 1.d0)

    dtn_dx = 2 * phi * x / (1 + x**2 + y**2)**2
    dtn_dy = 2 * phi * y / (1 + x**2 + y**2)**2

    dm(1,1) = ux * (cos(tn) - (x * sin(tn) + y * cos(tn)) * dtn_dx)
    dm(1,2) = ux * (-sin(tn) - (x * sin(tn) + y * cos(tn)) * dtn_dy)
    dm(2,1) = uy * (sin(tn) + (x * cos(tn) - y * sin(tn)) * dtn_dx)
    dm(2,2) = uy * (cos(tn) + (x * cos(tn) - y * sin(tn)) * dtn_dy)

    return
  end subroutine jacobian_matrix

  subroutine calculate_distance(x1, y1, x2, y2, distance) 
    real(8), intent(in) :: x1, y1, x2, y2
    real(8), intent(out) :: distance
    distance = sqrt((x2 - x1)**2 + (y2 - y1)**2)
  end subroutine calculate_distance

  subroutine find_fixed_points(x_fp, y_problema, y_fp_NR)
    use constantes
    implicit none
    real(8), intent(in) :: x_fp, y_problema
    real(8), intent(out) :: y_fp_NR
    integer :: j, root_count, current_root, status_B
    real(8) :: y_0, func_value, y_star, highest_root, residuo
    real(8), dimension(grid_y, 2) :: signs
    real(8), allocatable :: root_brackets(:,:)
    real(8), dimension(2,2) :: dm

    root_count = 0
    highest_root = -1.0d99
  
    !$omp parallel do default(firstprivate) shared(signs)
    loop_y: do j = 1, grid_y
      y_0 = y_min + (y_max - y_min) * float(j - 1) / float(grid_y - 1)
      call func_fp(y_0, func_value)
      !$omp critical
      signs(j,1) = y_0
      signs(j,2) = func_value
      !$omp end critical
    end do loop_y
    !$omp end parallel do
  
    do j = 1, grid_y - 1
      if ((signs(j,2)*signs(j+1,2)) < 0.d0) then
        if ((y_problema > signs(j,1)) .and. (y_problema < signs(j+1,1))) cycle
        if ((-y_problema > signs(j,1)) .and. (-y_problema < signs(j+1,1))) cycle
        root_count = root_count + 1
      end if
    end do
  
    allocate(root_brackets(root_count,2))
  
    current_root = 0
    do j = 1, grid_y - 1
      if ((signs(j,2)*signs(j+1,2)) < 0.d0) then
        if ((y_problema > signs(j,1)) .and. (y_problema < signs(j+1,1))) cycle
        if ((-y_problema > signs(j,1)) .and. (-y_problema < signs(j+1,1))) cycle
        current_root = current_root + 1
        root_brackets(current_root, 1) = signs(j, 1)
        root_brackets(current_root, 2) = signs(j+1, 1)
      end if
    end do
  
    do j = 1, root_count
      call bisection(root_brackets(j,1), root_brackets(j,2), y_star, status_B)
      if (status_B == 0) then
        if (y_star > highest_root) highest_root = y_star
        call jacobian_matrix(dm, x_fp, y_star)
        residuo = (2.d0 - (dm(1,1) + dm(2,2))) / 4.d0
        if ((residuo < 1.d0) .and. (residuo > 0.d0)) then
          ! write(13,*) 0.5d0, y_star
        else
          ! write(14,*) 0.5d0, y_star
        end if
      end if
    end do
  
    y_fp_NR = highest_root
  end subroutine find_fixed_points   
  
  subroutine sort_three_numbers(numbers)
    implicit none
    integer, intent(inout) :: numbers(3)
    integer :: temp

    ! verifica se o array contem 3 num distintos
    if (numbers(1) == numbers(2) .or. numbers(2) == numbers(3) .or. numbers(1) == numbers(3)) then
      print*, "Error: The array must contain three distinct integers."
      print*, numbers
      return
    end if

    ! ordena os numeros usando comparacoes
    if (numbers(1) > numbers(2)) then
      temp = numbers(1)
      numbers(1) = numbers(2)
      numbers(2) = temp
    end if
    if (numbers(2) > numbers(3)) then
      temp = numbers(2)
      numbers(2) = numbers(3)
      numbers(3) = temp
    end if
    if (numbers(1) > numbers(2)) then
      temp = numbers(1)
      numbers(1) = numbers(2)
      numbers(2) = temp
    end if

  end subroutine sort_three_numbers  
  
  subroutine func_fp(y_0, func_value)
    use constantes
    implicit none
    real(8), intent(in) :: y_0
    real(8), intent(out) :: func_value
    real(8) :: t_n, left_hand_side, right_hand_side

    t_n = theta - phi / (1.25d0 + y_0**2)
    left_hand_side = y_0
    right_hand_side = (cos(t_n) + 1.0d0) / (2.0d0 * sin(t_n))
    func_value = left_hand_side - right_hand_side

  end subroutine func_fp
  
  subroutine bisection(a_in, b_in, root, status)
    use constantes
    implicit none

    real(8), intent(in) :: a_in, b_in   
    real(8), intent(out) :: root  
    integer, intent(out) :: status ! Status (0: success, 1: failure)
    real(8) :: fa, fb, c, fc, func_value, a, b
    integer :: iter

    a = a_in
    b = b_in

    ! Check if the initial interval is valid
    call func_fp(a, func_value)
    fa = func_value
    call func_fp(b, func_value)
    fb = func_value
    
    if (fa * fb > 0.0) then
      print *, "Bissection Error: f(a) and f(b) must have opposite signs."
      status = 1
      return
    end if

    ! Initialize
    iter = 0
    status = 1

    do while (iter < max_iter)
      c = (a + b) / 2.0   
      call func_fp(c, func_value)
      fc = func_value

      ! Check for convergence
      if (abs(fc) <= tol .or. abs(b - a) / 2.0 <= tol) then
        root = c     
        status = 0
        return
      end if

      ! Update the interval
      if (fa * fc < 0.0) then
        b = c
        fb = fc
      else
        a = c
        fa = fc
      end if

      iter = iter + 1
    end do

    print *, "Bissection Warning: Maximum number of iterations reached."
    root = c
    status = 1
  end subroutine bisection 
    
  subroutine slater_condition(x, y, status)
    use constantes
    implicit none
    real(8), intent(in) :: x, y
    integer, intent(out) :: status ! Status (0: satisfeito com 3 tempos, 1: nao satisfeito, 2: 2 tempos distintos, 3: menos de 2 tempos)
    
    real(8) :: x0, y0
    real(8) :: x1, y1, x_ref, y_ref, distance
    integer :: i, previous_rec, rec_count, rec_time, quarta_rec
    integer, dimension(3) :: rec_times 
 
      x0 = x; y0 = y
       
      rec_times = -1
      previous_rec = 0
      rec_count = 0
      quarta_rec = -1

      x_ref = x0; y_ref = y0
      
      loop_iter: do i = 1, 100000
        call mapa(x0, y0, x1, y1)
        x0=x1;y0=y1

       ! calculo de tempos de retorno             
        if (rec_count < 4) then
        
        call calculate_distance(x_ref, y_ref, x1, y1, distance)            
        
          if (distance < eps) then
            rec_time = i - previous_rec

            if (rec_count == 0) then
              rec_times(1) = rec_time
              rec_count = rec_count + 1

            else if ((rec_count == 1) .and. (rec_times(1) /= rec_time)) then
              rec_times(2) = rec_time
              rec_count = rec_count + 1

            else if ((rec_count == 2) .and. (rec_times(1) /= rec_time) .and. (rec_times(2) /= rec_time)) then
              rec_times(3) = rec_time
              rec_count = rec_count + 1
              
            else if ((rec_count == 3) .and. (rec_times(1) /= rec_time) .and. (rec_times(2) /= rec_time) .and. &
              & (rec_times(3) /= rec_time)) then
              quarta_rec = rec_time
              exit loop_iter
            end if

          previous_rec = i
          end if    
        
        end if
        !!!!!!!!      
      end do loop_iter     

      if (rec_count == 3) then
        call sort_three_numbers(rec_times)
        if ( ( (rec_times(1) + rec_times(2)) /= rec_times(3) ) .or. (quarta_rec .ne. -1)) then
          status = 1            ! se os 3 tempos violam slater ou foi medido um quarto tempo
        else  
          status = 0            ! 3 tempos medidos satisfazem slater
        end if
      elseif (rec_count == 2) then
        status = 2             ! 2 tempos distintos encontrados
      else
        status = 3             ! menos de 2 tempos encontrados  
      end if     

    return
  end subroutine slater_condition
    
  function weighting_function(x)
    implicit none
    real(8), intent(in) :: x
    real(8) :: weighting_function
    if (x > 0 .and. x< 1) then
      weighting_function = exp(-1.d0/(x*(1.d0-x)))
    else
      weighting_function = 0
    end if
  end function weighting_function

  subroutine central_difference_derivative(data, ndata, delta_x, derivative)
    implicit none
    integer, intent(in) :: ndata
    real(8), intent(in) :: data(ndata)
    real(8), intent(in) :: delta_x
    real(8), intent(out) :: derivative(ndata)
    integer :: i

    do i = 2, ndata - 1
      derivative(i) = (data(i + 1) - data(i - 1)) / (2.0d0 * delta_x)
    end do
  
    ! Use mirroring scheme for the first point
    derivative(1) = (data(2) - data(1)) / delta_x
    ! Use mirroring scheme for the last point
    derivative(ndata) = (data(ndata) - data(ndata - 1)) / delta_x
  
  end subroutine central_difference_derivative

  subroutine continuity_check(x0, y0, x_fp, y_fp, continuity_status) 
    use constantes
    implicit none
    real(8), intent(in) :: x0, y0, x_fp, y_fp
    integer, intent(out) :: continuity_status ! 0: continuo; 1: descontinuo

    real(8) :: x_0, y_0, x_1, y_1, theta_c, tol_discontinuity
    real(8), allocatable :: orbit(:), deriv(:), smoothed_orbit(:)
    integer :: i, iter
  
    iter = 10000

    allocate(orbit(iter)); allocate(deriv(iter)); allocate(smoothed_orbit(iter))

    x_0 = x0; y_0 = y0

    do i = 1, iter
      call mapa(x_0, y_0, x_1, y_1)
      x_0 = x_1
      y_0 = y_1
      theta_c = atan2(y_0 - y_fp, x_0 - x_fp)
      if (theta_c < 0.d0) theta_c = theta_c + 2.d0*pi
      orbit(i) = theta_c
    end do

    call sort_orbit(orbit)

    if (maxval(orbit) - minval(orbit) < 6.d0) then
      continuity_status = 1
      return
    end if  

    call central_difference_derivative(orbit, iter, 1.0d0, deriv)
    deriv(1) = deriv(2); deriv(iter) = deriv(iter-1)  
    tol_discontinuity = 0.03d0

    do i = 2, iter   
      if (abs(deriv(i) - deriv(i-1)) > tol_discontinuity) then
        continuity_status = 1
        return
      else
        continuity_status = 0
      end if
    end do

  end subroutine continuity_check

  subroutine sort_orbit(orbit)
    implicit none
    real(8), intent(inout) :: orbit(:)
    integer :: i, j, n
    real(8) :: temp
  
    n = size(orbit)
    do i = 1, n-1
      do j = i+1, n
        if (orbit(i) > orbit(j)) then
          temp = orbit(i)
          orbit(i) = orbit(j)
          orbit(j) = temp
        end if
      end do
    end do
  
  end subroutine sort_orbit

  subroutine rot_num_profile_superconvergent_y(y_fp, x_fp, y_i, y_f, rot_num, num_iterations, num_points, chaos)
    use constantes
    implicit none
    real(8), intent(in) :: y_f, y_fp, x_fp, y_i
    real(8), dimension(:,:), allocatable, intent(out) :: rot_num
    integer, dimension(:), allocatable :: chaos                         !0: não violou slater; 1: violou slater
    integer, intent(in) :: num_iterations, num_points
    real(8) :: y, x0, y0, theta_i, theta_f, theta_c0, theta_c1
    real(8) :: x1, y1, delta_theta, x_ref, y_ref, distance, soma_denominador, soma_omega
    integer :: j, i, previous_rec, rec_count, rec_time, k, quarta_rec
    real(8), dimension(:), allocatable :: orbit
    integer, dimension(3) :: rec_times     ! identificar ICs que nao satisfazem criterio de Slater
                                           ! atribuir a elas um valor alto de w para nao serem identificadas como min
  
    integer :: total_iter ! usado para calcular rot num com menos iter quando slater é violado

    allocate(rot_num(num_points, 2))
    allocate(chaos(num_points))
  
    chaos = 0

   !$omp parallel do private(x_ref, y_ref, y, x0, y0, theta_i, theta_f, theta_c0, theta_c1, x1, y1, delta_theta, &
    !$omp&                   i, distance, soma_denominador, soma_omega, orbit) &
    !$omp& firstprivate(rec_times, rec_count, previous_rec, rec_time, total_iter, quarta_rec) schedule(dynamic)
    loop_ic: do j = 1, num_points     
  
        y = y_i + (y_f - y_i) * float(j - 1) / float(num_points - 1)   
  
        x0 = x_fp
        y0 = y    
  
        x_ref = x0; y_ref = y0
  
        theta_i = atan2(y0 - y_fp, x0 - x_fp)
        theta_f = theta_i
        theta_c0 = theta_i
  
        rec_times = -1
        previous_rec = 0
        rec_count = 0 
        
        allocate(orbit(num_iterations+1))

        total_iter = num_iterations

        orbit(1) = theta_i

        loop_iter: do i = 1, num_iterations
          call mapa(x0, y0, x1, y1)
          x0=x1;y0=y1
          
          theta_c1 = atan2(y0 - y_fp, x0 - x_fp)        
          if (theta_c1 < 0.d0) theta_c1 = theta_c1 + 2.d0*pi        
          delta_theta = theta_c1 - theta_c0
          delta_theta = mod(delta_theta, 2.d0*pi)       
          if (delta_theta < 0.d0) delta_theta = delta_theta + 2.d0*pi               
          theta_f = theta_f + delta_theta        
          theta_c0 = theta_c1

          orbit(i+1) = theta_f 

          !!!!!!!! calculo de tempos de retorno ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~            
          if (rec_count < 4) then
            call calculate_distance(x_ref, y_ref, x1, y1, distance)            
              if (distance < eps) then
                rec_time = i - previous_rec
                if (rec_count == 0) then
                  rec_times(1) = rec_time
                  rec_count = rec_count + 1
                else if ((rec_count == 1) .and. (rec_times(1) /= rec_time)) then
                  rec_times(2) = rec_time
                  rec_count = rec_count + 1
                else if ((rec_count == 2) .and. (rec_times(1) /= rec_time) .and. (rec_times(2) /= rec_time)) then
                  rec_times(3) = rec_time
                  rec_count = rec_count + 1                  
                  !checar se slater é violado, se sim, parar de iterar
                  call sort_three_numbers(rec_times)
                  if ( ( (rec_times(1) + rec_times(2)) /= rec_times(3) ) ) then
                    chaos(j) = 1
                    total_iter = i
                    exit loop_iter
                  end if  
                else if ((rec_count == 3) .and. (rec_times(1) /= rec_time) .and. (rec_times(2) /= rec_time) &
                  & .and. (rec_times(3) /= rec_time)) then
                  quarta_rec = rec_time
                  !aqui, foi encontrada uma quarta recorrencia e o criterio de slater foi violado. parar de iterar
                  total_iter = i
                  chaos(j) = 1
                  exit loop_iter
                end if
              previous_rec = i
              end if    
            end if
          !~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~          

        end do loop_iter

        soma_denominador = 0.d0
        do k = 1, total_iter+1 
          soma_denominador = soma_denominador + weighting_function(dble(k-1)/dble(total_iter+1))
        end do  
        soma_omega = 0.d0
        do i = 1, total_iter+1
          soma_omega=soma_omega+weighting_function(dble(i-1)/dble(total_iter+1))*(orbit(i+1)-orbit(i))/soma_denominador
        end do
        rot_num(j,1) = y; rot_num(j,2) = soma_omega/(2.d0*pi)

        deallocate(orbit)
  
    end do loop_ic
    !$omp end parallel do
  
    return
  end subroutine rot_num_profile_superconvergent_y

  subroutine process_segment_y(start_seg, end_seg, deriv1, rot_num, num_extrema, y_fp)
    implicit none
    integer, intent(in) :: start_seg, end_seg
    real(8), intent(in) :: deriv1(:)
    real(8), intent(in) :: rot_num(:, :), y_fp
    integer, intent(out) :: num_extrema
    real(8), parameter :: pi = acos(-1.d0)
    logical :: isolated
    integer :: i, j, continuity_status, slater_status, iter
    real(8) :: x0, y0, x1, y1

    if (end_seg-start_seg < 10) then
      return
    end if

    num_extrema = 0
  
    ! identificar trocas de sinal na derivada
    do i = start_seg + 1, end_seg - 1
      if (deriv1(i) * deriv1(i-1) < 0.d0) then
        isolated = .true.

        ! filtrar trocas isoladas
        do j = max(i-15, start_seg+1), min(i+15, end_seg-1)
          if (j /= i) then
            if (deriv1(j) * deriv1(j-1) < 0.d0) then
              isolated = .false.
              exit
            end if
          end if
        end do

        if (isolated) then

          x0 = 0.5d0
          y0 = rot_num(i, 1)

          call slater_condition(x0, y0, slater_status)
          call continuity_check(x0, y0, 0.5d0, y_fp, continuity_status)

          if (continuity_status == 0 .and. slater_status /= 1) then
            num_extrema = num_extrema + 1
            ! write(20,*) rot_num(i, 1), rot_num(i, 2)
            ! ! curva sem shear
            ! do iter = 1, 10000
            !   call mapa(x0, y0, x1, y1)
            !   write(3,*) x1, y1
            !   x0 = x1
            !   y0 = y1
            ! end do

          end if

         end if
      end if
    end do

  end subroutine process_segment_y

end module subrotinas
  
  
